/**
 * HRM Reports Module - JavaScript
 */

(function ($) {
    'use strict';

    const HRMReports = {
        currentReportType: null,
        currentReportData: null,

        init: function () {
            this.bindEvents();
            this.setDefaultDates();
        },

        bindEvents: function () {
            // Report type selection
            $('.select-report-btn').on('click', this.handleReportSelection.bind(this));

            // Close config panel
            $('.close-config-btn').on('click', this.closeConfigPanel.bind(this));

            // Generate report
            $('#report-config-form').on('submit', this.generateReport.bind(this));

            // Export buttons
            $('#export-excel').on('click', this.exportToExcel.bind(this));
            $('#export-pdf').on('click', this.exportToPDF.bind(this));
            $('#print-report').on('click', this.printReport.bind(this));

            // Refresh button
            $('#refresh-reports').on('click', this.refreshPage.bind(this));

            // Quick employee filter sync
            $('#quick-employee-filter').on('change', this.handleQuickFilterChange.bind(this));
        },

        setDefaultDates: function () {
            const today = new Date();
            const firstDay = new Date(today.getFullYear(), today.getMonth(), 1);

            $('#date-to').val(this.formatDate(today));
            $('#date-from').val(this.formatDate(firstDay));
        },

        formatDate: function (date) {
            const year = date.getFullYear();
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const day = String(date.getDate()).padStart(2, '0');
            return `${year}-${month}-${day}`;
        },

        handleReportSelection: function (e) {
            e.preventDefault();
            const card = $(e.target).closest('.report-type-card');
            const reportType = card.data('report');

            this.currentReportType = reportType;

            // Update config panel title
            const titles = {
                'attendance': 'Configure Attendance Report',
                'performance': 'Configure Performance Report',
                'employee': 'Configure Employee Report',
                'tasks': 'Configure Tasks Report',
                'daily-progress': 'Configure Daily Progress Report',
                'leave': 'Configure Leave Report',
                'summary': 'Configure Executive Summary'
            };

            $('#config-title').text(titles[reportType] || 'Configure Report');
            $('#report-type').val(reportType);

            // Sync quick filter to config panel
            const quickFilterValue = $('#quick-employee-filter').val();
            $('#employee-filter').val(quickFilterValue);

            // Show/hide relevant fields
            if (reportType === 'tasks') {
                $('#status-select-row').show();
            } else {
                $('#status-select-row').hide();
            }

            // Show/hide leave type filter
            if (reportType === 'leave') {
                $('#leave-type-select-row').show();
            } else {
                $('#leave-type-select-row').hide();
            }

            if (reportType === 'summary') {
                $('#employee-select-row').hide();
            } else {
                $('#employee-select-row').show();
            }

            // Show config panel with animation
            $('#report-config-panel').slideDown(300);

            // Scroll to config panel
            $('html, body').animate({
                scrollTop: $('#report-config-panel').offset().top - 100
            }, 500);
        },

        handleQuickFilterChange: function (e) {
            // Sync quick filter with config panel employee filter
            const selectedEmployee = $(e.target).val();
            $('#employee-filter').val(selectedEmployee);
        },

        closeConfigPanel: function (e) {
            e.preventDefault();
            $('#report-config-panel').slideUp(300);
            $('#report-results').slideUp(300);
        },

        generateReport: function (e) {
            e.preventDefault();

            const formData = {
                action: 'hrm_generate_report',
                nonce: hrm_admin_params.nonce,
                report_type: $('#report-type').val(),
                date_from: $('#date-from').val(),
                date_to: $('#date-to').val(),
                employee_id: $('#employee-filter').val(),
                status: $('#status-filter').val(),
                leave_type: $('#leave-type-filter').val(),
                leave_status: $('#leave-status-filter').val()
            };

            // Show loading state
            this.showLoading();

            $.ajax({
                url: hrm_admin_params.ajaxurl,
                type: 'POST',
                data: formData,
                success: (response) => {
                    this.hideLoading();
                    if (response.success) {
                        this.currentReportData = response.data;
                        this.displayReport(response.data);
                    } else {
                        this.showError('Failed to generate report. Please try again.');
                    }
                },
                error: () => {
                    this.hideLoading();
                    this.showError('An error occurred. Please try again.');
                }
            });
        },

        showLoading: function () {
            const loadingHTML = `
                <div class="report-loading">
                    <div class="loading-spinner"></div>
                    <div class="loading-text">Generating report...</div>
                </div>
            `;

            $('#results-content').html(loadingHTML);
            $('#report-results').slideDown(300);

            $('html, body').animate({
                scrollTop: $('#report-results').offset().top - 100
            }, 500);
        },

        hideLoading: function () {
            $('.report-loading').fadeOut(200);
        },

        displayReport: function (data) {
            $('#results-title').text(data.title);
            $('#results-subtitle').text(data.subtitle);

            let contentHTML = '';

            switch (data.type) {
                case 'attendance':
                    contentHTML = this.renderAttendanceReport(data);
                    break;
                case 'performance':
                    contentHTML = this.renderPerformanceReport(data);
                    break;
                case 'employee':
                    contentHTML = this.renderEmployeeReport(data);
                    break;
                case 'tasks':
                    contentHTML = this.renderTasksReport(data);
                    break;
                case 'daily-progress':
                    contentHTML = this.renderDailyProgressReport(data);
                    break;
                case 'leave':
                    contentHTML = this.renderLeaveReport(data);
                    break;
                case 'summary':
                    contentHTML = this.renderSummaryReport(data);
                    break;
            }

            $('#results-content').html(contentHTML);
            $('#report-results').slideDown(300);

            // Scroll to results
            $('html, body').animate({
                scrollTop: $('#report-results').offset().top - 100
            }, 500);

            // Add animation to cards
            this.animateCards();
        },

        renderAttendanceReport: function (data) {
            let html = '<div class="report-stats-grid">';

            html += `
                <div class="report-stat-card">
                    <h4>Total Records</h4>
                    <div class="report-stat-value">${data.stats.total_records}</div>
                    <div class="report-stat-label">Attendance entries</div>
                </div>
                <div class="report-stat-card">
                    <h4>Present</h4>
                    <div class="report-stat-value">${data.stats.present}</div>
                    <div class="report-stat-label">${data.stats.attendance_rate}% rate</div>
                </div>
                <div class="report-stat-card">
                    <h4>Absent</h4>
                    <div class="report-stat-value">${data.stats.absent}</div>
                    <div class="report-stat-label">Days absent</div>
                </div>
                <div class="report-stat-card">
                    <h4>Total Hours</h4>
                    <div class="report-stat-value">${data.stats.total_hours}</div>
                    <div class="report-stat-label">Avg: ${data.stats.average_hours}h</div>
                </div>
            `;

            html += '</div>';

            // Chart
            if (data.chart_data) {
                html += this.renderPieChart(data.chart_data, 'Attendance Distribution');
            }

            // Table
            if (data.records && data.records.length > 0) {
                html += '<div class="report-table-wrapper"><table class="report-table">';
                html += '<thead><tr><th>Employee</th><th>Date</th><th>Check In</th><th>Check Out</th><th>Hours</th><th>Status</th></tr></thead>';
                html += '<tbody>';

                data.records.forEach(record => {
                    html += `
                        <tr>
                            <td>${record.employee_name || 'Unknown'}</td>
                            <td>${this.formatDisplayDate(record.date)}</td>
                            <td>${record.check_in ? this.formatTime(record.check_in) : '-'}</td>
                            <td>${record.check_out ? this.formatTime(record.check_out) : '-'}</td>
                            <td>${record.hours_worked}h</td>
                            <td><span class="status-badge status-${record.status}">${this.formatStatus(record.status)}</span></td>
                        </tr>
                    `;
                });

                html += '</tbody></table></div>';
            } else {
                html += this.renderEmptyState('No attendance records found');
            }

            return html;
        },

        renderPerformanceReport: function (data) {
            let html = '<div class="report-stats-grid">';

            html += `
                <div class="report-stat-card">
                    <h4>Total Tasks</h4>
                    <div class="report-stat-value">${data.stats.total_tasks}</div>
                    <div class="report-stat-label">Assigned tasks</div>
                </div>
                <div class="report-stat-card">
                    <h4>Completed</h4>
                    <div class="report-stat-value">${data.stats.completed}</div>
                    <div class="report-stat-label">${data.stats.completion_rate}% rate</div>
                </div>
                <div class="report-stat-card">
                    <h4>In Progress</h4>
                    <div class="report-stat-value">${data.stats.in_progress}</div>
                    <div class="report-stat-label">Active tasks</div>
                </div>
                <div class="report-stat-card">
                    <h4>Pending</h4>
                    <div class="report-stat-value">${data.stats.pending}</div>
                    <div class="report-stat-label">Not started</div>
                </div>
            `;

            html += '</div>';

            // Chart
            if (data.chart_data) {
                html += this.renderPieChart(data.chart_data, 'Task Status Distribution');
            }

            // Progress Stats Table
            if (data.progress_stats && data.progress_stats.length > 0) {
                html += '<h3 style="margin: 30px 0 20px; font-size: 20px; color: #1a202c;">Employee Performance</h3>';
                html += '<div class="report-table-wrapper"><table class="report-table">';
                html += '<thead><tr><th>Employee</th><th>Total Entries</th><th>Total Hours</th><th>Avg Hours/Entry</th></tr></thead>';
                html += '<tbody>';

                data.progress_stats.forEach(stat => {
                    html += `
                        <tr>
                            <td>${stat.employee_name || 'Unknown'}</td>
                            <td>${stat.total_entries}</td>
                            <td>${parseFloat(stat.total_hours || 0).toFixed(2)}h</td>
                            <td>${parseFloat(stat.avg_hours || 0).toFixed(2)}h</td>
                        </tr>
                    `;
                });

                html += '</tbody></table></div>';
            }

            return html;
        },

        renderEmployeeReport: function (data) {
            let html = `
                <div class="report-stats-grid">
                    <div class="report-stat-card">
                        <h4>Total Employees</h4>
                        <div class="report-stat-value">${data.total_employees}</div>
                        <div class="report-stat-label">In this report</div>
                    </div>
                </div>
            `;

            if (data.employees && data.employees.length > 0) {
                html += '<div class="report-table-wrapper"><table class="report-table">';
                html += '<thead><tr><th>Name</th><th>Email</th><th>Present Days</th><th>Total Hours</th><th>Tasks</th><th>Completed</th><th>Progress Entries</th></tr></thead>';
                html += '<tbody>';

                data.employees.forEach(emp => {
                    const attendance = emp.attendance || {};
                    const tasks = emp.tasks || {};

                    html += `
                        <tr>
                            <td>${emp.employee.display_name}</td>
                            <td>${emp.employee.user_email}</td>
                            <td>${attendance.present_days || 0}/${attendance.total_days || 0}</td>
                            <td>${parseFloat(attendance.total_hours || 0).toFixed(2)}h</td>
                            <td>${tasks.total_tasks || 0}</td>
                            <td>${tasks.completed_tasks || 0}</td>
                            <td>${emp.progress_entries || 0}</td>
                        </tr>
                    `;
                });

                html += '</tbody></table></div>';
            } else {
                html += this.renderEmptyState('No employee data found');
            }

            return html;
        },

        renderTasksReport: function (data) {
            let html = '<div class="report-stats-grid">';

            html += `
                <div class="report-stat-card">
                    <h4>Total Tasks</h4>
                    <div class="report-stat-value">${data.stats.total_tasks}</div>
                    <div class="report-stat-label">All tasks</div>
                </div>
                <div class="report-stat-card">
                    <h4>Completed</h4>
                    <div class="report-stat-value">${data.stats.completed}</div>
                    <div class="report-stat-label">${data.stats.completion_rate}% rate</div>
                </div>
                <div class="report-stat-card">
                    <h4>In Progress</h4>
                    <div class="report-stat-value">${data.stats.in_progress}</div>
                    <div class="report-stat-label">Active</div>
                </div>
                <div class="report-stat-card">
                    <h4>Pending</h4>
                    <div class="report-stat-value">${data.stats.pending}</div>
                    <div class="report-stat-label">Not started</div>
                </div>
            `;

            html += '</div>';

            // Chart
            if (data.chart_data) {
                html += this.renderPieChart(data.chart_data, 'Task Status Overview');
            }

            // Tasks Table
            if (data.tasks && data.tasks.length > 0) {
                html += '<h3 style="margin: 30px 0 20px; font-size: 20px; color: #1a202c;">Task Details</h3>';
                html += '<div class="report-table-wrapper"><table class="report-table">';
                html += '<thead><tr><th>Title</th><th>Assigned To</th><th>Due Date</th><th>Status</th><th>Created</th></tr></thead>';
                html += '<tbody>';

                data.tasks.forEach(task => {
                    html += `
                        <tr>
                            <td>${task.title}</td>
                            <td>${task.employee_name || 'Unassigned'}</td>
                            <td>${task.due_date ? this.formatDisplayDate(task.due_date) : 'No deadline'}</td>
                            <td><span class="status-badge status-${task.status.replace('_', '-')}">${this.formatStatus(task.status)}</span></td>
                            <td>${this.formatDisplayDate(task.created_at)}</td>
                        </tr>
                    `;
                });

                html += '</tbody></table></div>';
            } else {
                html += this.renderEmptyState('No tasks found');
            }

            return html;
        },

        renderDailyProgressReport: function (data) {
            let html = '<div class="report-stats-grid">';

            html += `
                <div class="report-stat-card">
                    <h4>Total Entries</h4>
                    <div class="report-stat-value">${data.stats.total_entries}</div>
                    <div class="report-stat-label">Progress updates</div>
                </div>
                <div class="report-stat-card">
                    <h4>Total Hours</h4>
                    <div class="report-stat-value">${data.stats.total_hours}</div>
                    <div class="report-stat-label">Logged hours</div>
                </div>
                <div class="report-stat-card">
                    <h4>Completed</h4>
                    <div class="report-stat-value">${data.stats.completed}</div>
                    <div class="report-stat-label">Finished tasks</div>
                </div>
                <div class="report-stat-card">
                    <h4>Average Hours</h4>
                    <div class="report-stat-value">${data.stats.average_hours}</div>
                    <div class="report-stat-label">Per entry</div>
                </div>
            `;

            html += '</div>';

            // Chart
            if (data.chart_data) {
                html += this.renderPieChart(data.chart_data, 'Progress Status');
            }

            // Progress Table
            if (data.entries && data.entries.length > 0) {
                html += '<h3 style="margin: 30px 0 20px; font-size: 20px; color: #1a202c;">Progress Entries</h3>';
                html += '<div class="report-table-wrapper"><table class="report-table">';
                html += '<thead><tr><th>Employee</th><th>Date</th><th>Task</th><th>Hours</th><th>Description</th><th>Status</th></tr></thead>';
                html += '<tbody>';

                data.entries.forEach(entry => {
                    const description = entry.description.length > 100
                        ? entry.description.substring(0, 100) + '...'
                        : entry.description;

                    html += `
                        <tr>
                            <td>${entry.employee_name || 'Unknown'}</td>
                            <td>${this.formatDisplayDate(entry.date)}</td>
                            <td>${entry.task_title || 'General'}</td>
                            <td>${entry.hours_logged}h</td>
                            <td>${this.stripHtml(description)}</td>
                            <td><span class="status-badge status-${entry.status.replace('_', '-')}">${this.formatStatus(entry.status)}</span></td>
                        </tr>
                    `;
                });

                html += '</tbody></table></div>';
            } else {
                html += this.renderEmptyState('No progress entries found');
            }

            return html;
        },

        renderLeaveReport: function (data) {
            let html = '<div class="report-stats-grid">';

            html += `
                <div class="report-stat-card">
                    <h4>Total Requests</h4>
                    <div class="report-stat-value">${data.stats.total_requests}</div>
                    <div class="report-stat-label">Leave applications</div>
                </div>
                <div class="report-stat-card">
                    <h4>Approved</h4>
                    <div class="report-stat-value" style="color: #10B981;">${data.stats.approved}</div>
                    <div class="report-stat-label">${data.stats.approval_rate}% approval rate</div>
                </div>
                <div class="report-stat-card">
                    <h4>Pending</h4>
                    <div class="report-stat-value" style="color: #F59E0B;">${data.stats.pending}</div>
                    <div class="report-stat-label">Awaiting review</div>
                </div>
                <div class="report-stat-card">
                    <h4>Rejected</h4>
                    <div class="report-stat-value" style="color: #EF4444;">${data.stats.rejected}</div>
                    <div class="report-stat-label">Declined requests</div>
                </div>
                <div class="report-stat-card">
                    <h4>Total Leave Days</h4>
                    <div class="report-stat-value">${data.stats.total_leave_days}</div>
                    <div class="report-stat-label">Days taken/requested</div>
                </div>
                <div class="report-stat-card">
                    <h4>Employees</h4>
                    <div class="report-stat-value">${data.stats.unique_employees}</div>
                    <div class="report-stat-label">Unique requesters</div>
                </div>
            `;

            html += '</div>';

            // Status Chart
            if (data.chart_data) {
                html += this.renderLeaveStatusChart(data.chart_data, 'Leave Status Distribution');
            }

            // Leave by Type Chart
            if (data.leave_by_type) {
                html += this.renderLeaveTypeChart(data.leave_by_type, 'Leave by Type');
            }

            // Leave Records Table
            if (data.records && data.records.length > 0) {
                html += '<h3 style="margin: 30px 0 20px; font-size: 20px; color: #1a202c;">Leave Records</h3>';
                html += '<div class="report-table-wrapper"><table class="report-table">';
                html += '<thead><tr><th>Employee</th><th>Leave Type</th><th>Date Range</th><th>Days</th><th>Reason</th><th>Status</th><th>Applied On</th></tr></thead>';
                html += '<tbody>';

                data.records.forEach(record => {
                    const statusClass = `status-leave-${record.status}`;
                    const reason = record.reason && record.reason.length > 50
                        ? record.reason.substring(0, 50) + '...'
                        : (record.reason || 'N/A');

                    html += `
                        <tr>
                            <td><strong>${record.employee_name || 'Unknown'}</strong></td>
                            <td><span class="leave-type-badge leave-type-${record.leave_type}">${this.formatLeaveType(record.leave_type)}</span></td>
                            <td>${this.formatDisplayDate(record.start_date)} → ${this.formatDisplayDate(record.end_date)}</td>
                            <td style="text-align: center;"><strong>${record.total_days}</strong></td>
                            <td>${this.stripHtml(reason)}</td>
                            <td><span class="status-badge ${statusClass}">${this.formatStatus(record.status)}</span></td>
                            <td>${this.formatDisplayDate(record.created_at)}</td>
                        </tr>
                    `;
                });

                html += '</tbody></table></div>';
            } else {
                html += this.renderEmptyState('No leave records found for the selected criteria');
            }

            return html;
        },

        renderLeaveStatusChart: function (data, title) {
            const total = Object.values(data).reduce((a, b) => a + b, 0);
            if (total === 0) return '';

            const colors = {
                'pending': '#F59E0B',
                'approved': '#10B981',
                'rejected': '#EF4444'
            };

            let html = '<div class="report-chart-container"><h3>' + title + '</h3>';
            html += '<div style="display: flex; gap: 20px; align-items: center; flex-wrap: wrap;">';

            Object.keys(data).forEach(key => {
                const percentage = ((data[key] / total) * 100).toFixed(1);
                const color = colors[key] || '#0C8776';

                html += `
                    <div style="flex: 1; min-width: 150px; text-align: center;">
                        <div style="width: 80px; height: 80px; margin: 0 auto; border-radius: 50%; background: ${color}; display: flex; align-items: center; justify-content: center; color: white; font-weight: bold; font-size: 18px;">
                            ${percentage}%
                        </div>
                        <div style="margin-top: 10px; font-weight: 600; text-transform: capitalize;">
                            ${key.replace('_', ' ')}
                        </div>
                        <div style="color: #718096; font-size: 14px;">
                            ${data[key]} requests
                        </div>
                    </div>
                `;
            });

            html += '</div></div>';
            return html;
        },

        renderLeaveTypeChart: function (data, title) {
            const total = Object.values(data).reduce((a, b) => a + b, 0);
            if (total === 0) return '';

            const colors = {
                'casual': '#3B82F6',
                'sick': '#EF4444',
                'annual': '#10B981',
                'unpaid': '#8B5CF6'
            };

            let html = '<div class="report-chart-container"><h3>' + title + '</h3>';
            html += '<div style="display: flex; gap: 20px; align-items: center; flex-wrap: wrap;">';

            Object.keys(data).forEach(key => {
                if (data[key] === 0) return; // Skip empty types

                const percentage = ((data[key] / total) * 100).toFixed(1);
                const color = colors[key] || '#0C8776';

                html += `
                    <div style="flex: 1; min-width: 150px; text-align: center;">
                        <div style="width: 80px; height: 80px; margin: 0 auto; border-radius: 50%; background: ${color}; display: flex; align-items: center; justify-content: center; color: white; font-weight: bold; font-size: 18px;">
                            ${percentage}%
                        </div>
                        <div style="margin-top: 10px; font-weight: 600; text-transform: capitalize;">
                            ${key} Leave
                        </div>
                        <div style="color: #718096; font-size: 14px;">
                            ${data[key]} requests
                        </div>
                    </div>
                `;
            });

            html += '</div></div>';
            return html;
        },

        formatLeaveType: function (type) {
            const types = {
                'casual': 'Casual',
                'sick': 'Sick',
                'annual': 'Annual',
                'unpaid': 'Unpaid'
            };
            return types[type] || type.charAt(0).toUpperCase() + type.slice(1);
        },

        renderSummaryReport: function (data) {
            let html = '';

            // Attendance Summary
            html += '<div class="report-chart-container">';
            html += '<h3>📊 Attendance Overview</h3>';
            html += '<div class="report-stats-grid">';
            html += `
                <div class="report-stat-card">
                    <h4>Present</h4>
                    <div class="report-stat-value">${data.attendance.present}</div>
                </div>
                <div class="report-stat-card">
                    <h4>Absent</h4>
                    <div class="report-stat-value">${data.attendance.absent}</div>
                </div>
                <div class="report-stat-card">
                    <h4>Total Hours</h4>
                    <div class="report-stat-value">${data.attendance.total_hours}</div>
                </div>
                <div class="report-stat-card">
                    <h4>Attendance Rate</h4>
                    <div class="report-stat-value">${data.attendance.attendance_rate}%</div>
                </div>
            `;
            html += '</div></div>';

            // Tasks Summary
            html += '<div class="report-chart-container">';
            html += '<h3>📋 Tasks Overview</h3>';
            html += '<div class="report-stats-grid">';
            html += `
                <div class="report-stat-card">
                    <h4>Total Tasks</h4>
                    <div class="report-stat-value">${data.tasks.total_tasks}</div>
                </div>
                <div class="report-stat-card">
                    <h4>Completed</h4>
                    <div class="report-stat-value">${data.tasks.completed}</div>
                </div>
                <div class="report-stat-card">
                    <h4>In Progress</h4>
                    <div class="report-stat-value">${data.tasks.in_progress}</div>
                </div>
                <div class="report-stat-card">
                    <h4>Completion Rate</h4>
                    <div class="report-stat-value">${data.tasks.completion_rate}%</div>
                </div>
            `;
            html += '</div></div>';

            // Progress Summary
            html += '<div class="report-chart-container">';
            html += '<h3>✅ Daily Progress Overview</h3>';
            html += '<div class="report-stats-grid">';
            html += `
                <div class="report-stat-card">
                    <h4>Total Entries</h4>
                    <div class="report-stat-value">${data.progress.total_entries}</div>
                </div>
                <div class="report-stat-card">
                    <h4>Total Hours</h4>
                    <div class="report-stat-value">${data.progress.total_hours}</div>
                </div>
                <div class="report-stat-card">
                    <h4>Completed</h4>
                    <div class="report-stat-value">${data.progress.completed}</div>
                </div>
                <div class="report-stat-card">
                    <h4>Average Hours</h4>
                    <div class="report-stat-value">${data.progress.average_hours}</div>
                </div>
            `;
            html += '</div></div>';

            // Top Performers
            if (data.top_performers && data.top_performers.length > 0) {
                html += '<div class="report-chart-container">';
                html += '<h3>🏆 Top Performers</h3>';
                html += '<div class="report-table-wrapper"><table class="report-table">';
                html += '<thead><tr><th>Rank</th><th>Employee</th><th>Completed Tasks</th><th>Total Hours</th></tr></thead>';
                html += '<tbody>';

                data.top_performers.forEach((performer, index) => {
                    html += `
                        <tr>
                            <td><strong>#${index + 1}</strong></td>
                            <td>${performer.employee_name || 'Unknown'}</td>
                            <td>${performer.completed_tasks}</td>
                            <td>${parseFloat(performer.total_hours || 0).toFixed(2)}h</td>
                        </tr>
                    `;
                });

                html += '</tbody></table></div></div>';
            }

            return html;
        },

        renderPieChart: function (data, title) {
            // Simple visual representation using CSS
            const total = Object.values(data).reduce((a, b) => a + b, 0);
            if (total === 0) return '';

            let html = '<div class="report-chart-container"><h3>' + title + '</h3>';
            html += '<div style="display: flex; gap: 20px; align-items: center; flex-wrap: wrap;">';

            const colors = {
                'present': '#48bb78',
                'absent': '#f56565',
                'late': '#ed8936',
                'completed': '#4299e1',
                'in_progress': '#ecc94b',
                'pending': '#a0aec0'
            };

            Object.keys(data).forEach(key => {
                const percentage = ((data[key] / total) * 100).toFixed(1);
                const color = colors[key] || '#667eea';

                html += `
                    <div style="flex: 1; min-width: 150px; text-align: center;">
                        <div style="width: 80px; height: 80px; margin: 0 auto; border-radius: 50%; background: ${color}; display: flex; align-items: center; justify-content: center; color: white; font-weight: bold; font-size: 18px;">
                            ${percentage}%
                        </div>
                        <div style="margin-top: 10px; font-weight: 600; text-transform: capitalize;">
                            ${key.replace('_', ' ')}
                        </div>
                        <div style="color: #718096; font-size: 14px;">
                            ${data[key]} items
                        </div>
                    </div>
                `;
            });

            html += '</div></div>';
            return html;
        },

        renderEmptyState: function (message) {
            return `
                <div class="report-empty-state">
                    <span class="dashicons dashicons-clipboard"></span>
                    <h3>No Data Found</h3>
                    <p>${message}</p>
                </div>
            `;
        },

        animateCards: function () {
            $('.report-stat-card').each(function (index) {
                $(this).css({
                    opacity: 0,
                    transform: 'translateY(20px)'
                }).delay(index * 50).animate({
                    opacity: 1
                }, 300).css({
                    transform: 'translateY(0)'
                });
            });
        },

        exportToExcel: function (e) {
            e.preventDefault();

            if (!this.currentReportData) {
                alert('No report data to export');
                return;
            }

            const formData = {
                action: 'hrm_export_report',
                nonce: hrm_admin_params.nonce,
                report_type: this.currentReportData.type,
                format: 'excel',
                date_from: $('#date-from').val(),
                date_to: $('#date-to').val(),
                employee_id: $('#employee-filter').val(),
                status: $('#status-filter').val(),
                leave_type: $('#leave-type-filter').val(),
                leave_status: $('#leave-status-filter').val()
            };

            // Create a form and submit it
            const form = $('<form>', {
                method: 'POST',
                action: hrm_admin_params.ajaxurl
            });

            $.each(formData, function (key, value) {
                form.append($('<input>', {
                    type: 'hidden',
                    name: key,
                    value: value
                }));
            });

            form.appendTo('body').submit().remove();
        },

        exportToPDF: function (e) {
            e.preventDefault();

            if (!this.currentReportData) {
                alert('No report data to export');
                return;
            }

            const formData = {
                action: 'hrm_export_report',
                nonce: hrm_admin_params.nonce,
                report_type: this.currentReportData.type,
                format: 'pdf',
                date_from: $('#date-from').val(),
                date_to: $('#date-to').val(),
                employee_id: $('#employee-filter').val(),
                status: $('#status-filter').val(),
                leave_type: $('#leave-type-filter').val(),
                leave_status: $('#leave-status-filter').val()
            };

            // Create a form and submit it
            const form = $('<form>', {
                method: 'POST',
                action: hrm_admin_params.ajaxurl,
                target: '_blank'
            });

            $.each(formData, function (key, value) {
                form.append($('<input>', {
                    type: 'hidden',
                    name: key,
                    value: value
                }));
            });

            form.appendTo('body').submit().remove();
        },

        printReport: function (e) {
            e.preventDefault();
            window.print();
        },

        refreshPage: function (e) {
            e.preventDefault();
            location.reload();
        },

        formatDisplayDate: function (dateString) {
            if (!dateString) return '-';
            const date = new Date(dateString);
            return date.toLocaleDateString('en-US', {
                year: 'numeric',
                month: 'short',
                day: 'numeric'
            });
        },

        formatTime: function (timeString) {
            if (!timeString) return '-';
            const date = new Date(timeString);
            return date.toLocaleTimeString('en-US', {
                hour: '2-digit',
                minute: '2-digit'
            });
        },

        formatStatus: function (status) {
            return status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase());
        },

        stripHtml: function (html) {
            const tmp = document.createElement('DIV');
            tmp.innerHTML = html;
            return tmp.textContent || tmp.innerText || '';
        },

        showError: function (message) {
            alert(message);
        }
    };

    // Initialize when document is ready
    $(document).ready(function () {
        HRMReports.init();
    });

})(jQuery);
